/************************************************************************
 * NAME:	ti.c
 *
 * DESCR:	Implements the t99 format.
 *
 * NOTES:	two different formats are implemented here
 *		   v9t9 - raw sectors - small images (called t99 in this code)
 *		   pc99 - complete track data - big images hard to decode
 *	 	anadisk - NOT implemented here
 ************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>

#include "standard.h"
#include "floppy.h"
#include "error.h"

/************************************************************************
 * NAME:	t99_report()
 *
 * DESCR:	Generates a report of the (apparently) t99 file.
 *
 * ARGS:	level refers to the amount of reporting, 1 is lowest,
 *		3 is highest.
 *
 * RETURNS:	
 *
 * NOTES:	- levels other than 1 cause a detailed look at the file
 *		  beyond the guessing that occurred before
 ************************************************************************/
int
t99_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("T99");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = t99_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

/************************************************************************
 * NAME:	pc99_report()
 *
 * DESCR:	Generates a report of the (apparently) pc99 file.
 *
 * ARGS:	level refers to the amount of reporting, 1 is lowest,
 *		3 is highest.
 *
 * RETURNS:	
 *
 * NOTES:	- levels other than 1 cause a detailed look at the file
 *		  beyond the guessing that occurred before
 ************************************************************************/
int
pc99_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("PC99");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
	return;
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = pc99_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

#define T99_SECTOR_SIZE	256
#define T99_DD_SECTORS	18
#define T99_SD_SECTORS	9
#define T99_TRACKS	40

#define T99_DD_SIZE	(T99_SECTOR_SIZE * T99_TRACKS * T99_DD_SECTORS)
#define T99_SD_SIZE	(T99_SECTOR_SIZE * T99_TRACKS * T99_SD_SECTORS)

#define T99_DD_MARK	0xfb
#define T99_SD_MARK	0xfb

/************************************************************************
 * NAME:	t99_guesser()
 *
 * DESCR:	Tries to guess a t99 format.  This is a very simple format.
 *		The code basically looks to see if it is 40 tracks by
 *		9 sectors by 256 bytes.  
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- 40 * 9 * 256 = 36 * 10 * 256 = 92160
 *		- which means a 36 track image with 10 sectors fakes this out
 ************************************************************************/
int
t99_guesser(int fd)
{
    struct stat		statbuf;
    encoding		apparent_encoding;

    fstat(fd,&statbuf);

    /* t99 format must have 40 * 9 * 256 bytes	*/
    /*   or 40 * 18 * 256 for double density	*/

    if (statbuf.st_size == T99_SD_SIZE ) {
	apparent_encoding = WD_FM;	/* apparently single density	*/
	return(TRUE);
    } else if (statbuf.st_size == T99_DD_SIZE ) {
	apparent_encoding = WD_MFM;	/* apparently double density	*/
	return(TRUE);
    }

    return(FALSE);
}

#define PC99_SECTOR_SIZE	256
#define PC99_TRACKS		40

#define PC99_DD_SECTORS		18
#define PC99_SD_SECTORS		9

#define PC99_DD_TRACK_LEN	6872
#define PC99_SD_TRACK_LEN	3253

#define PC99_SSSD_FILE_SIZE	(PC99_SD_TRACK_LEN*PC99_TRACKS)
#define PC99_DSSD_FILE_SIZE	(2*PC99_SSSD_FILE_SIZE)
#define PC99_SSDD_FILE_SIZE	(PC99_DD_TRACK_LEN*PC99_TRACKS)
#define PC99_DSDD_FILE_SIZE	(2*PC99_SSDD_FILE_SIZE)

#define PC99_MAX_TRACK_LEN	MAX(PC99_DD_TRACK_LEN,PC99_SD_TRACK_LEN)

/************************************************************************
 * NAME:	pc99_size_check()
 *
 * DESCR:	Checks the size of the pc99 file and returns an indicator
 *		of its size.
 *
 * ARGS:	
 *
 * RETURNS:	0 - it doesn't look like a pc99
 *		1 - it is a SSSD
 *		2 - it is a DSSD
 *		3 - it is a SSDD
 *		4 - it is a DSDD
 *
 * NOTES:	
 ************************************************************************/
int
pc99_size_check(int fd)
{
    struct stat		statbuf;

    fstat(fd,&statbuf);

    switch (statbuf.st_size) {
        case PC99_SSSD_FILE_SIZE:	return(1);

        case PC99_DSSD_FILE_SIZE:	return(2);

        case PC99_SSDD_FILE_SIZE:	return(3);

        case PC99_DSDD_FILE_SIZE:	return(4);

        default:			return(0);
    }
}

/************************************************************************
 * NAME:	pc99_format_check()
 *
 * DESCR:	Given a file, check to see what format it looks like it
 *		is in:  DD or SD.
 *
 * ARGS:	
 *
 * RETURNS:	0 - doesn't appear to be a PC99
 *		1 - SD
 *		2 - DD
 *
 * NOTES:	
 ************************************************************************/
int
pc99_format_check(int fd)
{
#define SAMPLE_SIZE	100
#define MIN_SD_ZEROS	10
#define MIN_DD_4ES	10
#define MIN_DD_ZEROS	8
#define DD_A1S		3

    unsigned char	buffer[SAMPLE_SIZE];
    int			i,j;

    if (read(fd,buffer,SAMPLE_SIZE) != SAMPLE_SIZE) {
	return(FALSE);
    }

    lseek(fd,(off_t)0,SEEK_SET);	/* and rewind the file		*/

    switch (buffer[0]) {
        case 0x00:	/* should be SD	*/
	    /* look for 0x00's followed by a 0xfe	*/
	    /* must be at least MIN_ZEROS of zeros	*/
	    
	    for (i=0; i < SAMPLE_SIZE; i++) {
		if (buffer[i] != 0x00) {
		    break;
		}
	    }
	    if (i == SAMPLE_SIZE || buffer[i] != 0xfe || i < MIN_SD_ZEROS) {
		return(0);
	    }
	    return(1);

        case 0x4e:	/* should be DD	*/
	    /* look for 0x4e's following by 0x00 followed by 3 0xa1	*/
	    /* if all of THAT is right, then we're there!		*/

	    for (i=0; i < SAMPLE_SIZE; i++) {
		if (buffer[i] != 0x4e) {
		    break;
		}
	    }
	    if (i == SAMPLE_SIZE || buffer[i] != 0x00 || i < MIN_DD_4ES) {
		return(FALSE);
	    }

	    for (j = 0; j < SAMPLE_SIZE-i; j++) {
		if (buffer[j+i] != 0x00) {
		    break;
		}
	    }
	    if (j + i == SAMPLE_SIZE || j < MIN_DD_ZEROS) {
		return(FALSE);
	    }

	    if (buffer[j+i] != 0xa1 || buffer[j+i+1] != 0xa1 || buffer[j+i+2] != 0xa1) {
		return(FALSE);
	    }
	    return(TRUE);

        default:
	    return(0);
    }
}

/************************************************************************
 * NAME:	pc99_guesser()
 *
 * DESCR:	Tries to guess a pc99 format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- very simply looks at the file length and if it is
 *		  the right size, then it checks the front end to see
 *		  if it looks right too.
 ************************************************************************/
int
pc99_guesser(int fd)
{

    if (pc99_size_check(fd) == 0 || pc99_format_check(fd) == 0) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	t99_read()
 *
 * DESCR:	Read in the t99 format.  Pretty simple...just a bunch
 *		of tracks.  t99 can be either SD or DD.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int t99_read(int fd, struct floppy *floppy)
{
    int		track;
    int		sector;
    struct stat	statbuf;

    fstat(fd,&statbuf);

    if (statbuf.st_size == T99_DD_SIZE) {
	floppy->encoding = WD_MFM;		/* apparently double density	*/
    } else if (statbuf.st_size == T99_SD_SIZE ) {
	floppy->encoding = WD_FM;		/* apparently single density	*/
    } else {
	floppy->encoding = UNKNOWN_ENCODING;
    }

    floppy->write_protect = 0;		/* not write-protected		*/
    floppy->sides = 1;
    floppy->sectors = (floppy->encoding==WD_FM)?T99_SD_SECTORS:T99_DD_SECTORS;
    floppy->tracks = statbuf.st_size / T99_SECTOR_SIZE / floppy->sectors;

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

    for (track = 0; track < floppy->tracks; track++) {
	floppy->side[0][track].sector = 
	    (struct sector *)malloc(floppy->sectors*sizeof(struct sector));
	floppy->side[0][track].sectors = floppy->sectors;
    }

    for(track =0; track < floppy->tracks; track++) {
	int	count;

	for (sector = 0; sector < floppy->sectors; sector++) {

	    count = read(fd,floppy->side[0][track].sector[sector].data,(size_t)T99_SECTOR_SIZE);

	    if (count < T99_SECTOR_SIZE) {
		break;	/* must be done, didn't get a complete track*/
	    }

	    floppy->side[0][track].sector[sector].id = track;
	    floppy->side[0][track].sector[sector].side = 0;
	    floppy->side[0][track].sector[sector].sector = sector;
	    floppy->side[0][track].sector[sector].sizecode = 1;
	    floppy->side[0][track].sector[sector].size = 256;
	    floppy->side[0][track].sector[sector].mark = (floppy->encoding==WD_FM)?T99_SD_MARK:T99_DD_MARK;
	    floppy->side[0][track].sector[sector].headCRC = 0;
	    floppy->side[0][track].sector[sector].dataCRC = 0;
	    floppy->side[0][track].sector[sector].encoding = floppy->encoding;
	}

	if (count < T99_SECTOR_SIZE)
	    break; 	/* must be done	*/
    }

    floppy->tracks = track;

/********
/* don't know if we need this for ti99 or not...
/*
/*    dirtrack = 17; 	/* lock it in for now - may need to be	*/
/*			/* different for different encodings	*/
/*
/*    /* put directory marks in the directory track	*/
/*
/*    for (sector = 0; sector < floppy->sectors; sector++) {
/*	floppy->side[0][dirtrack].sector[sector].mark = (floppy->encoding==WD_FM)?0xfa:0xf8;
/*    }
******/

    /* need to do real errors	*/

    floppy_crc_set(floppy);	/* set the CRCs	*/

    return(0);
}



/************************************************************************
 * NAME:	pc99_find_address()
 *
 * DESCR:	Given a buffer of data, assumed to be WD, either FM or
 *		MFM, read the sector information for it.
 *
 * ARGS:	input - a pointer to input data
 *		end - an index to the end of the input data (EXCLUSIVE)
 *		sector - the sector structure is filled in with header data
 *
 * RETURNS:	the amount of data consumed
 *
 * NOTES:
 ************************************************************************/
int
pc99_find_address_sd(unsigned char	*input,
		     int		 end,
		     struct sector	*sector)
{
    int			start = -1;
    int			i;
    int			state = 0;

#define WD_SECTOR_ADDR_MARK	0xfe

    /* we assume that we are at the beginning of a track/sector			*/
    /* and we are pointing to leading zeros before the sector addres mark	*/

    for (i=0; i < end; i++) {

	/* note that we MUST start this state machine with a zero, so it will	*/
	/* skip all of the leading 0xff's that may be there (if any)		*/

	switch(state) {
	    case 0:	/* fall thru */	/* must have at least 6 zeros		*/
	    case 1:	/* fall thru */
	    case 2:	/* fall thru */
	    case 3:	/* fall thru */
	    case 4:	/* fall thru */
	    case 5:	if (input[i] == 0x00) { state++; } else { state = 0; } break;
	    case 6:	if (input[i] == 0x00) break;	/* can get more of these	*/
		        if (input[i] == WD_SECTOR_ADDR_MARK) { state++; } else { state = 0; } break;
	   default:	break;
	}

	if (state == 7) {		/* located a data address	*/
	    i++;			/* move to next byte		*/

	    if (end-i < 6) {
		break;			/* don't have enough data for header	*/
	    }

	    sector->id = input[i++];
	    sector->side = input[i++];
	    sector->sector = input[i++];
	    sector->sizecode = input[i++];

	    sector->headCRC = input[i++];
	    sector->headCRC |= input[i++] << 8;

	    break;
	}
    }

    return(i);
}

int
pc99_find_address_dd(unsigned char	*input,
		     int		 end,
		     struct sector	*sector)
{
    int			start = -1;
    int			i;
    int			state = 0;

#define WD_SECTOR_ADDR_MARK	0xfe

    /* we assume that we are at the beginning of a track/sector			*/
    /* and we are pointing to leading zeros before the sector addres mark	*/

    for (i=0; i < end; i++) {

	/* note that we MUST start this state machine with a zero, so it will	*/
	/* skip all of the leading 0xff's that may be there (if any)		*/

	switch(state) {
	    case 0:	/* fall thru */	/* must have at least 10 0x4e's		*/
	    case 1:	/* fall thru */
	    case 2:	/* fall thru */
	    case 3:	/* fall thru */
	    case 4:	/* fall thru */
	    case 5:	/* fall thru */
	    case 6:	/* fall thru */
	    case 7:	/* fall thru */
	    case 8:	/* fall thru */
	    case 9:	if (input[i] == 0x4e) { state++; } else { state = 0; } break;
	    case 10:	if (input[i] == 0x4e) break;	/* can get more of these	*/
	    case 11:	/* fall thru */ /* must have at least 6 0x00's		*/
	    case 12:	/* fall thru */
	    case 13:	/* fall thru */
	    case 14:	/* fall thru */
	    case 15:	if (input[i] == 0x00) { state++; } else { state = 0; } break;
	    case 16:	if (input[i] == 0x00) break;	/* can get more of these	*/
	    case 17:	/* fall thru */
	    case 18:	if (input[i] == 0xa1) { state++; } else { state = 0; } break;	
	    case 19:	if (input[i] == WD_SECTOR_ADDR_MARK) { state++; } else { state = 0; } break;

	    default:	break;
	}

	if (state == 20) {		/* located a data address	*/
	    i++;			/* move to next byte		*/

	    if (end-i < 6) {
		break;			/* don't have enough data for header	*/
	    }

	    sector->id = input[i++];
	    sector->side = input[i++];
	    sector->sector = input[i++];
	    sector->sizecode = input[i++];

	    sector->headCRC = input[i++];
	    sector->headCRC |= input[i++] << 8;

	    break;
	}
    }

    return(i);
}

/************************************************************************
 * NAME:	pc99_find_data()
 *
 * DESCR:	From the current position, find the sector data.
 *
 * ARGS:	
 *
 * RETURNS:	the index of the NEXT byte after the data was read, -1 if
 *			no data was found.
 *
 ************************************************************************/
int				       
pc99_find_data_sd(unsigned char	*input,
	       int		 end,
	       struct sector	*sector)
{
    int			i;
    int			state = 0;

    for (i=0; i < end; i++) {

	/* need at least 7 0xffs, followed by at least 6 0x00s, then	*/
	/* the sector mark, whatever that may be			*/

	switch(state) {
	    case 0:	/* fall thru */
	    case 1:	/* fall thru */
	    case 2:	/* fall thru */
	    case 3:	/* fall thru */
	    case 4:	/* fall thru */
	    case 5:	/* fall thru */
	    case 6:	/* fall thru */
	    case 7:	if (input[i] == 0xff) { state++; } else { state = 0; } break;
	    case 8:	if (input[i] != 0xff) { if (input[i] == 0x00) { state++; } else { state = 0;} } break;
	    case 9:	/* fall thru */
	    case 10:	/* fall thru */
	    case 11:	/* fall thru */
	    case 12:	/* fall thru */
	    case 13:	if (input[i] == 0x00) { state++; } else { state = 0; } break;
	    case 14:	if (input[i] != 0x00) { state++; } break;
	}

	if (state == 15) {		/* located the data mark	*/

	    sector->mark = input[i++];

	    if (end-i < 258) {
		break;			/* not enough data!	       	*/
	    }

	    memcpy(sector->data,&input[i],256);

	    i += 256;

	    sector->dataCRC = input[i++];
	    sector->dataCRC |= input[i++] << 8;

	    break;
	}
    }

    return(i);
}

int				       
pc99_find_data_dd(unsigned char	*input,
	       int		 end,
	       struct sector	*sector)
{
    int			i;
    int			state = 0;

    /* we assume that we are at the beginning of a track/sector			*/
    /* and we are pointing to leading zeros before the sector addres mark	*/

    for (i=0; i < end; i++) {

	/* note that we MUST start this state machine with a zero, so it will	*/
	/* skip all of the leading 0xff's that may be there (if any)		*/

	switch(state) {
	    case 0:	/* fall thru */	/* must have at least 10 0x4e's		*/
	    case 1:	/* fall thru */
	    case 2:	/* fall thru */
	    case 3:	/* fall thru */
	    case 4:	/* fall thru */
	    case 5:	/* fall thru */
	    case 6:	/* fall thru */
	    case 7:	/* fall thru */
	    case 8:	/* fall thru */
	    case 9:	if (input[i] == 0x4e) { state++; } else { state = 0; } break;
	    case 10:	if (input[i] == 0x4e) break;	/* can get more of these	*/
	    case 11:	/* fall thru */ /* must have at least 6 0x00's		*/
	    case 12:	/* fall thru */
	    case 13:	/* fall thru */
	    case 14:	/* fall thru */
	    case 15:	if (input[i] == 0x00) { state++; } else { state = 0; } break;
	    case 16:	if (input[i] == 0x00) break;	/* can get more of these	*/
	    case 17:	/* fall thru */
	    case 18:	if (input[i] == 0xa1) { state++; } else { state = 0; } break;	

	    default:	break;
	}

	if (state == 19) {		/* located the data mark	*/
	    i++;

	    sector->mark = input[i++];

	    if (end-i < 258) {
		break;			/* not enough data!	       	*/
	    }

	    memcpy(sector->data,&input[i],256);

	    i += 256;

	    sector->dataCRC = input[i++];
	    sector->dataCRC |= input[i++] << 8;

	    break;
	}
    }

    return(i);
}



/************************************************************************
 * NAME:	pc99_sector_read()
 *
 * DESCR:	Reads in a sector from the data buffer given in "ptr".
 *		Returns the amount of data consumed in ptr.
 *
 * ARGS:	ptr  - data buffer
 *		end  - index within ptr that is the end of data (exclusive)
 *
 * RETURNS:	-1 means that a sector couldn't be found
 *
 * NOTES:	
 ************************************************************************/
int
pc99_sector_read(unsigned char *ptr, int end, struct sector *sector, encoding encoding)
{
    int		 size1, size2;

    sector->encoding = encoding;
    sector->sizecode = 1;	/* locked at 256 byte sectors	*/
    sector->size = 256;

    if (encoding == WD_FM) {
	size1 = pc99_find_address_sd(ptr, end, sector);
    } else {
	size1 = pc99_find_address_dd(ptr, end, sector);
    }

    if (size1 <= 0) {
	return(-1);		/* couldn't find a sector	*/
    }

    ptr += size1;
    end -= size1;

    if (encoding == WD_FM) {
	size2 = pc99_find_data_sd(ptr, end, sector);
    } else {
	size2 = pc99_find_data_dd(ptr, end, sector);
    }

    if (size2 <= 0 ) {
	return(-1);			/* couldn't find data for sector	*/
    }

    return(size1+size2);
}


/************************************************************************
 * NAME:	pc99_track_read()
 *
 * DESCR:	Reads a given track from PC99 format.
 *
 * ARGS:	input - assumed to be a buffer of (size) data, alignment
 *			is such that the index hole as just passed.
 *		track - where the output data is to be written
 *
 * RETURNS:	the count of the number of sectors found, zero if an
 *		error of great proportion is discovered.
 *
 * NOTES:	- 
 ************************************************************************/
int
pc99_track_read(unsigned char *input, struct track *track, int size, encoding encoding)
{
    unsigned char	*ptr;
    int			 count;
    int			 sector;
    struct sector	 sector_buffer;

    ptr = input;

    track->sectors = 0;
    track->sector = NULL;

    for (sector=0;;sector++) {

	count = pc99_sector_read(ptr,size,&sector_buffer,encoding);

	if (count <= 0) {
	    break;				/* couldn't find a sector	*/
	}

	ptr += count;
	size -= count;

	track->sectors++;
	track->sector = (struct sector *)realloc(track->sector,track->sectors*sizeof(struct sector));
	track->sector[sector] = sector_buffer;
    }

    return(sector);
}



/************************************************************************
 * NAME:	pc99_read()
 *
 * DESCR:	Read in a pc99 style file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- as noted above, since the WD formats cannot be decoded
 *		  unless the "start of track" is given, this code assumes
 *		  that the start of the file is the same as the index
 *		  pulse (ie - the start of the track).
 ************************************************************************/
int
pc99_read(int fd, struct floppy *floppy)
{
    unsigned char	buffer[PC99_DD_TRACK_LEN];
    int			track, sectors;
    int			track_len;
    int			file_type;
    struct track	track_buffer;
    int			tracks_side[2];
    int			size;

    tracks_side[0] = 0;
    tracks_side[1] = 0;

    floppy->write_protect = 0;
    floppy->secsize = PC99_SECTOR_SIZE;
    floppy->tracks = PC99_TRACKS;

    file_type = pc99_size_check(fd);

    if (file_type == 0) {
	file_type = 1;		/* if forced, assume SSSD	*/
    }

    switch (file_type) {
        case 1:	/* SSSD	*/
	    floppy->encoding = WD_FM;
	    floppy->sectors = PC99_SD_SECTORS;
	    floppy->sides = 1;
	    break;

        case 2:	/* DSSD	*/
	    floppy->encoding = WD_FM;
	    floppy->sectors = PC99_SD_SECTORS;
	    floppy->sides = 2;
	    break;

        case 3: /* SSDD	*/
	    
	    floppy->encoding = WD_MFM;
	    floppy->sectors = PC99_DD_SECTORS;
	    floppy->sides = 1;
	    break;

        case 4: /* DSDD	*/

	    floppy->encoding = WD_MFM;
	    floppy->sectors = PC99_DD_SECTORS;
	    floppy->sides = 2;
	    break;
    }

    track_len = (floppy->encoding == WD_FM)?PC99_SD_TRACK_LEN:PC99_DD_TRACK_LEN;

    floppy->side[0] = NULL;
    floppy->side[1] = NULL;
    floppy->tracks = 0;

    for (;;) {		/* loop and loop */

	unsigned char	buffer[PC99_MAX_TRACK_LEN];
	int		side;

	size = read(fd,buffer,track_len);
	if (size != track_len) {			/* EOF reached	*/
	    if (size !=0) {
		M1("WARNING: EOF size read of %d (not zero)\n",size);
	    }
	    break;
	}

	sectors = pc99_track_read(buffer, &track_buffer, size, floppy->encoding);

	/* check the side number for the first sector, and use that to store	*/
	/* the sector in the floppy structure.					*/

	side = track_buffer.sector[0].side;

	tracks_side[side] += 1;
	floppy->side[side] = (struct track *)realloc(floppy->side[side], tracks_side[side]*sizeof(struct track));
	floppy->side[side][tracks_side[side]-1] = track_buffer;
	floppy->side[side][tracks_side[side]-1].sectors = sectors;
    }

    if (tracks_side[1] == 0) {
	floppy->sides = 1;
    } else {
	floppy->sides = 2;
    }

    floppy->sectors = floppy->side[0][0].sectors;
    floppy->tracks = MIN(tracks_side[0],tracks_side[1]);

    return(0);

}

/************************************************************************
 * NAME:	t99_dump()
 *
 * DESCR:	Write to the t99 format.  Pretty simple here too...note
 *		that a ton of information is lost when going to t99.  It
 *		only handles non-copy-protected disks, single sided.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	ERRORS are not checked.  Booooo,  hissss.
 ************************************************************************/
t99_dump(int fd,struct floppy *floppy )
{
    int	track;
    int sector;

    for (track = 0; track < floppy->tracks; track++) {
	for (sector = 0; sector < floppy->sectors; sector++) {
	    write(fd,LogicalSector(floppy,0,track,sector).data,256);
	}
    }
}

/************************************************************************
 * NAME:	pc99_dump()
 *
 * DESCR:	Dump out the pc99 format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- each track has to be an appropriate multiple of the
 *		  allowed track sizes.  Note that I have only seen DSSD
 *		  versions of this format...so all bets are off until I
 *		  se a DD version of some type.
 *			- 22 0x00 before first 0xfe
 *			- 0xfe
 *			- four bytes of header, 2 checksum
 *			- 11 0xff & 6 0x00 between header & 0xfb (DAM mark)
 *			- 0xfb
 *			- 2 checksum
 *			- 45 0xff & 6 0x00 between data an next header 0xfe
 *			- 276 0xffs at the end of the track
 *  (+ 22 (* 9 (+ 1 6 11 6 1 256 2)) (* 8 (+ 45 6)) 276)
 *		  (* 6506 20)
 ************************************************************************/
pc99_dump_sd(int fd,struct floppy *floppy )
{
#define SD_TRACK_START_ZEROS	22
#define SD_TRACK_END_FFS	276
#define SD_DATA_SEP_ZEROS	6
#define SD_DATA_SEP_FFS		11
#define SD_SECTOR_END_FFS	45
#define SD_SECTOR_END_ZEROS	6
#define SD_MAX_ZEROS		SD_TRACK_START_ZEROS
#define SD_MAX_FFS		SD_TRACK_END_FFS
#define ADDRESS_MARK		0xfe

    int	side, track, sector;
    int	i;
    unsigned char	zero_buffer[SD_MAX_ZEROS];
    unsigned char	ff_buffer[SD_MAX_FFS];
    unsigned char	buffer[10];

    for (i=0; i < SD_MAX_ZEROS; i++) {
	zero_buffer[i] = 0x00;
    }

    for (i=0; i < SD_MAX_FFS; i++) {
	ff_buffer[i] = 0xff;
    }

    for (side = 0; side < floppy->sides; side++) {
	for (track = 0; track < floppy->tracks; track++) {
	    write(fd,zero_buffer,SD_TRACK_START_ZEROS);
	    for (sector = 0; sector < floppy->sectors; sector++) {
		buffer[0] = ADDRESS_MARK;
		buffer[1] = floppy->side[side][track].sector[sector].id;
		buffer[2] = floppy->side[side][track].sector[sector].side;
		buffer[3] = floppy->side[side][track].sector[sector].sector;
		buffer[4] = 1;	/* size code - static at 256 */
		buffer[5] = floppy->side[side][track].sector[sector].headCRC%256;
		buffer[6] = floppy->side[side][track].sector[sector].headCRC/256;
		write(fd,buffer,7);
		write(fd,ff_buffer,SD_DATA_SEP_FFS);
		write(fd,zero_buffer,SD_DATA_SEP_ZEROS);
		buffer[0] = floppy->side[side][track].sector[sector].mark;
		write(fd,buffer,1);
		write(fd,floppy->side[side][track].sector[sector].data,256);
		buffer[0] = floppy->side[side][track].sector[sector].dataCRC%256;
		buffer[1] = floppy->side[side][track].sector[sector].dataCRC/256;
		write(fd,buffer,2);

		if (sector != floppy->sectors - 1) {
		    write(fd,ff_buffer,SD_SECTOR_END_FFS);
		    write(fd,zero_buffer,SD_SECTOR_END_ZEROS);
		}
	    }
	    write(fd,ff_buffer,SD_TRACK_END_FFS);
        }
    }
}
pc99_dump_dd(int fd,struct floppy *floppy )
{
#define DD_TRACK_START_4ES	16

#define DD_ADDRESS_4ES		24
#define DD_ADDRESS_ZEROS	10
#define DD_ADDRESS_A1S		3

#define DD_DATA_SEP_4ES		22
#define DD_DATA_SEP_ZEROS	12
#define DD_DATA_SEP_A1S		3

#define DD_TRACK_END_4ES	736

#define DD_MAX_ZEROS		DD_DATA_SEP_ZEROS
#define DD_MAX_4ES		DD_TRACK_END_4ES
#define DD_MAX_A1S		DD_DATA_SEP_A1S

    int	side, track, sector;
    int	i;
    unsigned char	zero_buffer[DD_MAX_ZEROS];
    unsigned char	foure_buffer[DD_MAX_4ES];
    unsigned char	a1_buffer[DD_MAX_A1S];
    unsigned char	buffer[10];

    for (i=0; i < DD_MAX_ZEROS; i++) {
	zero_buffer[i] = 0x00;
    }

    for (i=0; i < DD_MAX_4ES; i++) {
	foure_buffer[i] = 0x4e;
    }

    for (i=0; i < DD_MAX_A1S; i++) {
	a1_buffer[i] = 0xa1;
    }

    for (side = 0; side < floppy->sides; side++) {
	for (track = 0; track < floppy->tracks; track++) {
	    write(fd,foure_buffer,DD_TRACK_START_4ES);
	    for (sector = 0; sector < floppy->sectors; sector++)  {

		write(fd,foure_buffer,DD_ADDRESS_4ES);
		write(fd,zero_buffer,DD_ADDRESS_ZEROS);
		write(fd,a1_buffer,DD_ADDRESS_A1S);

		buffer[0] = ADDRESS_MARK;
		buffer[1] = floppy->side[side][track].sector[sector].id;
		buffer[2] = floppy->side[side][track].sector[sector].side;
		buffer[3] = floppy->side[side][track].sector[sector].sector;
		buffer[4] = 1;	/* size code - static at 256 */
		buffer[5] = floppy->side[side][track].sector[sector].headCRC%256;
		buffer[6] = floppy->side[side][track].sector[sector].headCRC/256;
		write(fd,buffer,7);

		write(fd,foure_buffer,DD_DATA_SEP_4ES);
		write(fd,zero_buffer,DD_DATA_SEP_ZEROS);
		write(fd,a1_buffer,DD_DATA_SEP_A1S);

		buffer[0] = floppy->side[side][track].sector[sector].mark;
		write(fd,buffer,1);

		write(fd,floppy->side[side][track].sector[sector].data,256);
		buffer[0] = floppy->side[side][track].sector[sector].dataCRC%256;
		buffer[1] = floppy->side[side][track].sector[sector].dataCRC/256;
		write(fd,buffer,2);
	    }
	    write(fd,foure_buffer,DD_TRACK_END_4ES);
        }
    }
}

pc99_dump(int fd,struct floppy *floppy )
{
    if (floppy->encoding == WD_FM) {
	pc99_dump_sd(fd,floppy);
    } else {
	pc99_dump_dd(fd,floppy);
    }
}


/************************************************************************
 * NAME:	pc99_init()
 *
 * DESCR:	Registers the pc99 floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
pc99_init()
{
    if (!floppy_format_register("PC99", "TI 99", pc99_guesser, pc99_read, pc99_dump, pc99_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("pc99", EXT_NOT_SPECIFIC, "PC99")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("p99", EXT_NOT_SPECIFIC, "PC99")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("p9", EXT_NOT_SPECIFIC, "PC99")) {
	return(FALSE);
    }

    /* this dsk format is extremely specific */

    if (!floppy_fileext_register("dsk", EXT_VERY_SPECIFIC, "PC99")) {
	return(FALSE);
    }
}

/************************************************************************
 * NAME:	t99_init()
 *
 * DESCR:	Registers the t99 floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the registration went OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
int
t99_init()
{
    if (!floppy_format_register("T99", "TI 99",t99_guesser, t99_read, t99_dump, t99_report)) {
	return(FALSE);
    }

    if (!floppy_fileext_register("v9t9", EXT_NOT_SPECIFIC, "T99")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("t99", EXT_NOT_SPECIFIC, "T99")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("t9", EXT_NOT_SPECIFIC, "T99")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("tidisk", EXT_NOT_SPECIFIC, "T99")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("dsk", EXT_NOT_SPECIFIC, "T99")) {
	return(FALSE);
    }

    return(TRUE);
}

